// Author: Sean Szumlanski
// Date: Summer 2025
//
// A basic weighted graph class with implementations of topological sort and
// Dijkstra's algorithm.


#include "weightedgraph.h"
#include "error.h"
#include "filelib.h"
#include "priorityqueue.h"
#include "queue.h"
#include <iostream>
using namespace std;

// This constructor expects a file with the following format:
//
// <number of nodes>
// <label for node 0>
// <label for node 1>
// ...
// <label for node (n-1)>
// <row 0 of adjacency matrix>
// <row 1 of adjacency matrix>
// ...
// <row (n-1) of adjacency matrix>
//
// For example, here is the input file for the graph from our slides on
// Dijkstra's algorithm (where a 0 represents the absence of an edge):
//
// 9
// A
// B
// C
// D
// E
// F
// G
// H
// I
// 0 5 0 0 0 0 9 18 1
// 5 0 7 0 0 0 0 0 0
// 0 7 0 8 11 0 0 0 6
// 0 0 8 0 20 0 0 0 0
// 0 0 11 20 0 4 0 0 2
// 0 0 0 0 4 0 1 0 0
// 9 0 0 0 0 1 0 2 3
// 18 0 0 0 0 0 2 0 0
// 1 0 6 0 2 0 3 0 0
//
// THIS FUNCTION HAS MINIMAL ERROR CHECKING TO DETERMINE WHETHER THE INPUT FILE
// IS FORMATTED CORRECTLY. However, the goal here was simply to write up a very
// basic graph class that would allow us to do some coding today, so:
// https://giphy.com/gifs/i-regret-nothing-chicken-11i77XHsbqUcZa
WeightedGraph::WeightedGraph(string filename)
{
    ifstream ifs;

    if (!openFile(ifs, filename))
    {
        error("Oh no! Failed to open \"" + filename + "\" in WeightedGraph()!");
    }

    string inputString;

    // Read number of nodes from first line of input file.
    ifs >> inputString;
    _numNodes = stringToInteger(inputString);
    _matrix.resize(_numNodes, _numNodes);

    // Read labels for each of our nodes.
    for (int i = 0; i < _numNodes; i++)
    {
        ifs >> inputString;
        _labels.add(inputString);

        if (_labelToIndexMap.containsKey(inputString))
        {
            error("Error: Duplicate label in WeightedGraph(): " + inputString);
        }
        _labelToIndexMap[inputString] = i;
    }

    // Read adjacency matrix from file. Note that a 0 in our input file
    // indicates a lack of edge.
    for (int i = 0; i < _numNodes; i++)
    {
        for (int j = 0; j < _numNodes; j++)
        {
            ifs >> inputString;
            _matrix[i][j] = stringToInteger(inputString);
        }
    }

    // Print graph's adjacency list.
    cout << "Created graph with " << _numNodes << " nodes." << endl << endl;

    for (int i = 0; i < _numNodes; i++)
    {
        cout << _labels[i] << ": ";

        for (int j = 0; j < _numNodes; j++)
        {
            if (_matrix[i][j])
            {
                cout << _labels[j] << "(" << _matrix[i][j] << ") ";
            }
        }

        cout << endl;
    }

    cout << endl;
}

WeightedGraph::~WeightedGraph()
{
    cout << "*poof* (a graph was just destroyed)" << endl;
}

// This is the priority-queue-based implementation of Dijkstra's algorithm.
// Recall that this approach is O(n^2 log n) in the worst case because of how
// we let trash build up in the priority queue. We can implement this in O(n^2)
// time by just looping through our dist[] array to find the smallest remaining
// value associated with an unvisited node at each step of the algorithm, but
// this approach is still interesting and worth exploring in code.
void WeightedGraph::printShortestPathsDijkstra(string startNodeLabel)
{
    Vector<int> dist(_numNodes, oo);
    Vector<bool> visited(_numNodes, false);
    Vector<int> source;

    // source[i] gives the index of the node that got us to i. Initially, we
    // set each node to be its own source. source[i] == i is used as the
    // terminating condition when we're following the breadcrumb trail backward
    // from some node during path reconstruction.
    for (int i = 0; i < _numNodes; i++)
    {
        source.add(i);
    }

    // If we can't start, we can't start.
    if (!_labelToIndexMap.containsKey(startNodeLabel))
    {
        error("Error: Could not find node in printShortestPaths(): " + startNodeLabel);
    }

    // Dijkstra's algorithm kicks off with the dist value at the source node set
    // to zero.
    int start = _labelToIndexMap[startNodeLabel];
    dist[start] = 0;

    PriorityQueue<int> pq;
    for (int i = 0; i < _numNodes; i++)
    {
        pq.enqueue(i, dist[i]);
    }

    // As long as there are nodes yet to be processed, keep going.
    while (!pq.isEmpty())
    {
        int thisOne = pq.dequeue();
        if (visited[thisOne])
        {
            continue;
        }

        // Update the dist[] values of all the neighbors of thisOne if this
        // unlocks a better (shorter) path to those nodes.
        for (int i = 0; i < _numNodes; i++)
        {
            // Recall that _matrix[i][j] == 0 means there is no edge. It would
            // be ruinous to eliminate this check and simply allow ourselves to
            // check whether dist[thisOne] + 0 < dist[i] in the case of non-
            // existent edges.
            if (_matrix[thisOne][i])
            {
                if (dist[thisOne] + _matrix[thisOne][i] < dist[i])
                {
                    // Update the cost of the shortest path to node i.
                    dist[i] = dist[thisOne] + _matrix[thisOne][i];

                    // Chuck this back into the priority queue with no regard
                    // for the trash that's accumulating there. I'm sure it'll
                    // be fine. :)
                    pq.enqueue(i, dist[i]);

                    // If thisOne unlocks a shorter path to i, keep track of
                    // that in our source vector.
                    source[i] = thisOne;
                }
            }
        }

        visited[thisOne] = true;
    }

    // Final output of all shortest paths (costs, as well as the actual paths
    // taken to get there).
    cout << "Shortest paths:" << endl << endl;

    for (int i = 0; i < _numNodes; i++)
    {
        cout << startNodeLabel << " to " << _labels[i] << " has cost " << dist[i] << ": ";
        printPath(i, source);
    }

    cout << endl;
}

// Follow the breadcrumb trail from the given node back to the starting node
// used to kick of Dijkstra's algorithm, and use nice formatting to print the
// path. Instead of using a stack, we could also implement this recursively.
void WeightedGraph::printPath(int i, Vector<int> source)
{
    Stack<int> path;

    while (source[i] != i)
    {
        path.push(i);
        i = source[i];
    }
    path.push(i);

    while (!path.isEmpty())
    {
        cout << _labels[path.pop()];
        if (!path.isEmpty())
        {
            cout << " -> ";
        }
    }
    cout << endl;
}

// Prints a topological sort for the given graph, if one exists. This also
// detects whether our graph contains a cycle.
void WeightedGraph::printTopologicalSort()
{
    // First, count the number of incoming edges at each node.
    Vector<int> incoming(_numNodes, 0);
    for (int i = 0; i < _numNodes; i++)
    {
        for (int j = 0; j < _numNodes; j++)
        {
            if (_matrix[i][j])
            {
                incoming[j]++;
            }
        }
    }

    // All the nodes with zero incoming edges are eligible to be added to our
    // topological sort.
    Queue<int> q;
    for (int i = 0; i < _numNodes; i++)
    {
        if (incoming[i] == 0)
        {
            q.enqueue(i);
        }
    }

    // Keep going as long as there are more nodes with zero incoming edges
    // in the queue, waiting to be added to our topological sort.
    Vector<int> sequence;
    while (!q.isEmpty())
    {
        int thisOne = q.dequeue();
        sequence.add(thisOne);

        // Now that we're adding this node to our sequence, all the nodes it
        // leads to (via a direct edge) have one less dependency.
        for (int i = 0; i < _numNodes; i++)
        {
            if (_matrix[thisOne][i])
            {
                incoming[i]--;

                // If removing this dependency means node i has no more incoming
                // edges, we can add it to our topological sort.
                if (incoming[i] == 0)
                {
                    q.enqueue(i);
                }
            }
        }
    }

    // A graph with a cycle has no valid topological sort (and vice versa, if
    // we cannot find a valid topological sort with this algorithm, that implies
    // it must contain a cycle).
    if (_numNodes != sequence.size())
    {
        cout << "Cycle detected. No valid topological sort." << endl;
        return;
    }

    // Print the sequence.
    cout << "Topological sort:" << endl;

    for (int i = 0; i < sequence.size(); i++)
    {
        cout << _labels[sequence[i]] << " ";
    }
    cout << endl << endl;
}
