"""
File: chat_server.py
--------------------
We will implement a simple chat server that handles
requests to get messages and add messages to our
running list of messages.
"""

from SimpleServer import run_server
import json

# Return of classes! :)
class ChatServer:
    def __init__(self):
        """
        Constructor
        -----------
        Initialize any instance variables needed for
        this class.
        """
        self.history = []

    def handle_request(self, request):
        """
        Method: handle_request
        -----------------------
        We've received a request from the internet!
        The request's command will either be:

            getMsgs
            newMsg

        This method will handle each command.
        """
        print(request)
        cmd = request.get_command()

        if cmd == "getMsgs":
            return self.get_msgs(request)
        elif cmd == "newMsg":
            return self.add_new_msg(request)
        else:
            return "Invalid command!"

    def get_msgs(self, request):
        """
        Method: get_msgs
        -----------------
        Returns a list of messages. The incoming request
        will come with an `index` field in the parameters
        dictionary that tells us which index we should
        slice at when returning the list of messages
        (i.e. I want all messages starting at index N).
        """
        cmd = request.get_command()
        if cmd != "getMsgs":
            return "Invalid command! Expected getMsgs!"

        params = request.get_params()
        if "index" not in params:
            return "Params not formatted correctly! Expected index as a key!"

        idx = int(params["index"])
        # Grab all messages starting from index and going to the end
        msgs = self.history[idx:]

        # Return the list data structure as a string
        return json.dumps(msgs)

    def add_new_msg(self, request):
        """
        Method: add_new_msg
        -------------------
        Adds a new message to our list of messages. This
        method also writes the latest message to history.json.
        Returns "success" assuming all is well.
        """

        # First step, add a new message to self.history
        cmd = request.get_command()
        if cmd != "newMsg":
            return "Invalid command! Was expecting newMsg!"

        params = request.get_params()
        if "msg" not in params:
            return "msg not included in params"

        if "user" not in params:
            return "user not included in params"

        user = params["user"]
        msg = params["msg"]

        # Store this message into self.history as follows:
        #     [user] msg
        formatted = f"[{user}] {msg}"
        self.history.append(formatted)

        # Last step: put this information into history.json
        with open("history.json", "w") as f:
            json.dump(self.history, f, indent=4)

        return "Success"

def main():
    # make the server
    handler = ChatServer()
    # start the server to handle internet requests!
    run_server(handler, 8080)

if __name__ == '__main__':
    main()