!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
! File solveLP.f90
!
!      program    solveLP
!
! Test program for reading an LP problem defined by dumpLP.m
! and solving it with Minos or quadMinos.
!
! Ding Ma and Michael Saunders, MS&E, Stanford University.
!
! In qminos56
!      make
! makes the Minos executables lib/libquadminos.a and lib/libquadminosdbg.a.
!
! In qminos56/testFBA
!      make solveLP
! compiles this program and links to the executables.
!
! 14 May 2012: First version of solveLP.f90 for SQOPT derived from sqtestLP.f90.
!              This program is specifically for reading and solving
!              an LP problem defined by file model_final_build_unscaled.mat
!              (with model.description = '01_TMA_ME_v1.0_maltose_minimal')
!              provided by Joshua Lerman, UC San Diego, Apr 2012.
!              Script dumpLP.m creates file TMA_ME_01.txt, which is
!              input by this program to recreate the LP problem.
!              m=18209, n=17535, ne=336301, cobj = e(17533).
!              The first 64-bit run (with default options) gave
!              Objective = 4.3816917628E-07 (min).  We don't know
!              the correct answer yet.
! 14 Jun 2012: The correct value seems to be 8.7036461682E-07
!              (obtained with quad precision with and without scaling).
! 06 Aug 2012: matlab/josh/Cobra/dumpLP.m now outputs cObj as last row of S.
!              The first TMA128 run as an LP
!              gave objective value 8.7036461686E-07 (11th digit different).
!
! 15 Apr 2014: solveLP.f90 for quadMINOS adapted from solveLP.f90 for SQOPT.
!              The input file (from dumpLP.m) assumes Ax + s = 0.
!              MINOS assumes the constraints are Ax - s = 0, so the
!              bounds on s are flipped here.
!
!              We bear in mind that the main data file is Double, not Quad.
!              Data arrays a,bl,bu are first read into Quad arrays, but then
!              internal subroutine qdq rounds them from Quad to Double and
!              back to Quad, so each element is of the form qa = [da 0].
!              This may increase the likelihood of very high precision solutions.
!
! 19 Apr 2014: minos56 and qminos56 files reorganized.
!              dumpLP.m modified to output problem name first.
!              The name is read here from the input file.
!+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

program solveLP

  implicit none

  integer, parameter        :: ip = 4, dp = 8, qp = 16

  integer(ip)               :: INFO, iExit
  integer(ip)               :: inLP, iSpecs, nnCon, nnJac, nnObj, nwcore
  integer(ip)               :: iObj, iPrint, iSumm, inform, nout
  integer(ip)               :: m, n, nb, ne, mincor, nInf, nname, nS
  real(qp)                  :: ObjAdd, Obj, sInf
  integer(ip),  allocatable :: hs(:)
  integer(ip),  allocatable :: ha(:), ka(:)
  real(qp),     allocatable :: a(:)
  real(qp),     allocatable :: bl(:), bu(:), cObj(:)
  real(qp),     allocatable :: xn(:), pi(:), rc(:)
  integer(ip),  allocatable :: name1(:), name2(:)
  character(8)              :: names(5), Probname

  integer(ip),  parameter   :: lenz = 5000000   ! As big as you like.
  real(qp)                 :: z(lenz)          ! This is the MINOS workspace.
  real(qp),     parameter   :: zero = 0.0_qp
  !------------------------------------------------------------------

  inLP     = 10        ! File from dumpLP.m (e.g. TMA_ME_01.txt)

  open(inLP, status='old')
  read(inLP,*) Probname! Problem name, up to 8 characters
  read(inLP,*) m       ! No of rows in A
  read(inLP,*) n       ! No of cols in A
  read(inLP,*) ne      ! No of nonzeros in A

  allocate( ka(n+1), a(ne), ha(ne) )
  read(inLP,*) ka      ! Pointers for A
  read(inLP,*) ha      ! Row indices
  read(inLP,*) a       ! Aij

  nb       = n + m
  nname    = 1
  nnCon    = 0
  nnJac    = 0
  nnObj    = 0
  ObjAdd   = zero
  iObj     = m

  allocate( bl(nb), bu(nb) )
  allocate( name1(nname), name2(nname) )
  allocate( hs(nb), pi(m), rc(nb), xn(nb) )

  read(inLP,*) bl        ! Lower bounds for x and slacks
  read(inLP,*) bu        ! Upper bounds for x and slacks
  close(inLP)

  ! The data (a,bl,bu) just read into Quad arrays was really just Double.
  ! For example, a(1) may now be [a11 a12] in Quad.
  ! Round from Quad to Double [a11] and then back to Quad as [a11 0]

  call qdq(  a, ne )
  call qdq( bl, nb )
  call qdq( bu, nb )

  ! Assign various names.
  ! These are relics from the days of MPS files.
  ! They appear in the MINOS Print file and/or Solution file.

  names(1) = Probname
  names(2) = 'c       '  ! Objective name
  names(3) = 'b       '  ! RHS name
  names(4) = '        '  ! Ranges name (bounds on slacks)
  names(5) = 'bounds  '  ! Bounds name (bounds on variables)

  ! Specify file numbers for MINOS.  (Others may be in the SPECS file.)
  ! 0 means that there should be no file.

  ispecs = 4   ! The MINOS SPECS   file.
! ispecs2= 5   ! The second SPECS  file (for warm start)
  iprint = 9   ! The MINOS PRINT   file.
  isumm  = 6   ! The MINOS SUMMARY file.
  nout   = 6   ! Local output file (6 = screen).

  ! Now we may open any number of files ourselves
  ! (perhaps to give them sensible names).
  ! For example:

  ! open( iprint, file='TMA_ME.out', status='UNKNOWN')
  ! open( ispecs, file='TMA_ME.spc', status='OLD')

  ! Alternatively, we may let mistart and minoss open them,
  ! using the method selected in subroutine m1open
  ! in the mi10*.f file that was used to build MINOS.
  !
  ! RULE OF THUMB:
  ! MINOS won't open file ispec (for example)
  ! if that unit is already open, or if ispec = 0.
  !------------------------------------------------------------------

  !------------------------------------------------------------------
  ! mistart MUST BE CALLED BEFORE ANY OTHER MINOS ROUTINE.
  !------------------------------------------------------------------
  call mistart( iprint, isumm, ispecs )  ! Initialize MINOS and open
                                        ! the specified files.

  call mispec( ispecs, inform )          ! Read the SPECS file
                                        ! (if ispecs > 0).

  if (inform >= 2) then
     write(nout, *) 'Error: ispecs > 0 but no SPECS file found'
     stop
  end if

  !----------------------------------------------------------------------
  ! Solve the problem.
  ! iobj   = 0    means there is no linear objective row in a(*).
  ! objadd = zero means there is no constant to be added to the objective.
  ! nname  = 1    means there are no meaningful names for the
  !               variables and constraints inside name1(*) and name2(*).
  !               MINOS will print default names.
  !----------------------------------------------------------------------
  hs(1:nb) = 0
  xn(1:nb) = zero
  pi(1:m)  = zero
  iobj     = m
  objadd   = zero
  nwcore   = lenz

  ! dumpLP.m generates slack bounds for SQOPT: Ax - s = 0
  ! Minos has Ax + s = 0, so we have to flip the bounds on s.

  z(1:m)     = -bl(n+1:nb)
  bl(n+1:nb) = -bu(n+1:nb)
  bu(n+1:nb) =   z(1:m)

  call minoss( 'Cold', m, n, nb, ne, nname,        &
               nncon, nnobj, nnjac,                &
               iobj, objadd, names,                &
               a, ha, ka, bl, bu, name1, name2,    &
               hs, xn, pi, rc,                     &
               inform, mincor, ns, ninf, sinf, obj, &
               z, nwcore )

  write(nout, *) ' '
  write(nout, *) 'Quad MINOS finished.'
  write(nout, *) 'inform =', inform
  write(nout, *) 'ninf   =', ninf
  write(nout, *) 'sinf   =', sinf
  write(nout, *) 'obj    =', obj

  !======================================================
  ! Warm start with 2nd SPECS file
  !======================================================
  call mispec( ispecs, inform )

  if (inform .ge. 2) then
     write(nout, *) 'No second SPECS file'
     go to 900
  end if

  ! 20 Apr 2014:
  ! Step 2 normally uses scaling (and then unscaling).
  ! The unscaling might fill in the 2nd half of each quad word.
  ! Round from Quad to Double and back to Quad again.

  call qdq(  a, ne )
  call qdq( bl, nb )
  call qdq( bu, nb )

  call minoss( 'Warm', m, n, nb, ne, nname,        &
               nncon, nnobj, nnjac,                &
               iobj, objadd, names,                &
               a, ha, ka, bl, bu, name1, name2,    &
               hs, xn, pi, rc,                     &
               inform, mincor, ns, ninf, sinf, obj, &
               z, nwcore )

  write(nout, *) ' '
  write(nout, *) 'Quad MINOS finished.'
  write(nout, *) 'inform =', inform
  write(nout, *) 'ninf   =', ninf
  write(nout, *) 'sinf   =', sinf
  write(nout, *) 'obj    =', obj

900 close( iprint )
  close( ispecs )

  deallocate( hs, pi, rc, xn )
  deallocate( name1, name2 )
  deallocate( bl, bu, ka, a, ha )

contains

  subroutine qdq( a, n )

    integer(ip), intent(in)    :: n
    real(qp),    intent(inout) :: a(n)

    ! 15 Apr 2014: First version of qdq.
    !              Round from Quad to Double and back to Quad.

    real(dp)                  :: da(n)   ! local array

    write(*,'(z32)') a(n)
    da = real( a,dp)
    a  = real(da,qp)
    write(*,'(z17)') da(n)
    write(*,'(z32)') a(n)
  end subroutine qdq

end program solveLP
