*     ------------------------------------------------------------------
*     File spring.f
*
*     This is a main program to generate an optimal control problem
*     of arbitrary size and solve it by calling MINOS as a subroutine.
*
*     The problem size depends on a parameter T.  There are
*     2T constraints and 3T + 2 variables, as well as bounds
*     on the variables.  The first T constraints are quadratic in
*     T + 1 variables, and the objective function is quadratic in
*     T + 1 other variables.
*
*     The control problem models a spring, mass and damper system.
*     It is of the form
*
*   --------------------------------------------------------------------
*   | minimize    1/2 sum x(t)**2   (t = 0 to T)                       |
*   |                                                                  |
*   | subject to                                                       |
*   |     y(t+1)  =  y(t)  -  0.01 y(t)**2  -  0.004 x(t)  +  0.2 u(t) |
*   |                                                                  |
*   |     x(t+1)  =  x(t)  +  0.2  y(t),                               |
*   |                                                                  |
*   |     y(t)   >=  -1,     -0.2  <=  u(t)  <=  0.2,                  |
*   |                                                                  |
*   |                (all for t = 0 to T-1)                            |
*   | and                                                              |
*   |     y(0)    =   0,      y(T)  =  0,       x(0) = 10.             |
*   --------------------------------------------------------------------
*
*     For large enough T (e.g. T >= 90), the optimal objective value
*     is about 1186.382.
*
*     This model with T = 100 was used as test problem 5.11 in
*     B. A. Murtagh and M. A. Saunders (1982), A projected Lagrangian
*     algorithm and its implementation for sparse nonlinear constraints,
*     Mathematical Programming Study 16, 84--117.
*
*     14 Nov 1994: First version of spring.f, derived from manne.f.
*     14 Sep 2000: Converted to use mistart.
*     ------------------------------------------------------------------

      program            spring

      implicit           real(8)          (a-h,o-z)

      parameter        ( maxT   = 2000,
     $                   maxm   = 2*maxT,
     $                   maxn   = 3*maxT + 2,
     $                   maxnb  = maxm + maxn,
     $                   maxne  = 7*maxT,
     $                   nname  = 1 )

      character*8        names(5)  , probnm
      integer            T
      integer            ha(maxne) , hs(maxnb)
      integer            ka(maxn+1), name1(nname), name2(nname)
      real(8)            a(maxne)  , bl(maxnb)   , bu(maxnb),
     $                   xn(maxnb) , pi(maxm)    , rc(maxnb)

      parameter          ( lenz = 1000000 )
      real(8)            z(lenz)

*     ------------------------------------------------------------------
*     The following MINOS common block allows access to nncon,
*     which defines T, the number of nonlinear constraints.
*     It is specified at runtime in the SPECS file.
*     ------------------------------------------------------------------
      common    /m8len / njac  ,nncon ,nncon0,nnjac
*     ------------------------------------------------------------------

*     Give names to the Problem, Objective, Rhs, Ranges and Bounds.

      names(1) = 'Spring..'
      names(2) = 'funobj  '
      names(3) = 'None    '
      names(4) = 'None    '
      names(5) = 'Bounds  '

!     Specify some file numbers.  (Others may be in the SPECS file.)
!     0 means that there should be no file.

      iprint = 9   ! The MINOS PRINT   file.
      isumm  = 6   ! The MINOS SUMMARY file.
      ispecs = 4   ! The MINOS SPECS   file.
      nout   = 6   ! Local output file (> 0).

!     ------------------------------------------------------------------
!     Now we may open any number of files ourselves
!     (perhaps to give them sensible names).
!     For example:
!
      open( iprint, file='spring.out', status='UNKNOWN')
      open( ispecs, file='spring.spc', status='OLD')
!
!     Alternatively, we may let mistart and minoss open them,
!     using the method selected in subroutine m1open
!     in the mi10*.f file that was used to build MINOS.
!
!     RULE OF THUMB:
!     MINOS won't open file ispec (for example)
!     if that unit is already open, or if ispec = 0.
!     ------------------------------------------------------------------


!     ------------------------------------------------------------------
!     mistart MUST BE CALLED BEFORE ANY OTHER MINOS ROUTINE.
!     ------------------------------------------------------------------
      call mistart( iprint, isumm, ispecs )  ! Initialize MINOS and open
                                             ! the specified files.


*     ------------------------------------------------------------------
*     Read a Specs file.  This must include "Nonlinear constraints  T"
*     for some integer T.
*     ------------------------------------------------------------------
      call mispec( ispecs, inform )

      if (inform .ge. 2) then
         write(nout, *) 'Error: ispecs > 0 but no SPECS file found'
         stop
      end if

      T     = nncon
      if (T .le. 1  .or.  T .gt. maxm/2) then
         write(nout, *) 'Invalid no. of Nonlinear constraints:', T
         stop
      end if

*     Write T into the problem name.

      write(probnm, '(i8)') T
      if      (T .lt.   100) then
         probnm(1:6) = 'Spring'
      else if (T .lt.  1000) then
         probnm(1:5) = 'Sprng'
      else if (T .lt. 10000) then
         probnm(1:4) = 'Spri'
      else
         probnm(1:3) = 'Spr'
      end if

      names(1) = probnm
      write(nout, *) 'Spring optimal control problem.    T =', T

*     ------------------------------------------------------------------
*     Generate a T-period problem.
*     ------------------------------------------------------------------
      call spdata( T, maxm, maxn, maxnb, maxne, inform,
     $             m, n, nb, ne, nncon, nnobj, nnjac,
     $             a, ha, ka, bl, bu, hs, xn, pi )

      if (inform .ge. 1) then
         write(nout, *) 'Not enough storage to generate a problem ',
     $                  'with  Nonlinear constraints =', T
         stop
      end if

*     ------------------------------------------------------------------
*     Specify options that were not set in the Specs file.
*     i1 and i2 may refer to the Print and Summary file respectively.
*     Setting them to 0 suppresses printing.
*     ------------------------------------------------------------------
      maxr   = T
      itnlim = T * 20
      i1     = 0
      i2     = 0
      call miopt ( 'Jacobian    Sparse',         i1, i2, inform )
      call miopti( 'Hessian dimension ', maxr  , i1, i2, inform )
      call miopti( 'Iterations        ', itnlim, i1, i2, inform )

*     ------------------------------------------------------------------
*     Go for it, using a Cold start.
*     iobj   = 0 means there is no linear objective row in a(*).
*     objadd = 0.0 means there is no constant to be added to the
*            objective.
*     hs     need not be set if a basis file is to be input.
*            Otherwise, each hs(1:n) should be 0, 1, 2, 3, 4, or 5.
*            The values are used by the Crash procedure
*            to choose an initial basis B.
*            If hs(j) = 0 or 1, column j is eligible for B.
*            If hs(j) = 2, column j is initially superbasic (not in B).
*            If hs(j) = 3, column j is eligible for B and is given
*                          preference over columns with hs(j) = 0 or 1.
*            If hs(j) = 4 or 5, column j is initially nonbasic.
*     ------------------------------------------------------------------
      iobj   = 0
      objadd = 0.0d+0
      nwcore = lenz

      call minoss( 'Cold', m, n, nb, ne, nname,
     $             nncon, nnobj, nnjac,
     $             iobj, objadd, names,
     $             a, ha, ka, bl, bu, name1, name2,
     $             hs, xn, pi, rc,
     $             inform, mincor, ns, ninf, sinf, obj,
     $             z, nwcore )

      if (inform .eq. 42) then
         write(nout, *) ' '
         write(nout, *) 'Estimate of required lenz:', mincor
         stop
      end if

      write(nout, *) ' '
      write(nout, *) 'minoss finished.'
      write(nout, *) 'inform =', inform
      if (inform .ge. 20) go to 900
*-->  Stop anyway
      if (inform .le. 20) stop
      stop

*     ------------------------------------------------------------------
*     Error exit.
*     ------------------------------------------------------------------
  900 write(nout, *) ' '
      write(nout, *) 'STOPPING because of error condition'
      stop

      end ! program spring

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine spdata( T, maxm, maxn, maxnb, maxne, inform,
     $                   m, n, nb, ne, nncon, nnobj, nnjac,
     $                   a, ha, ka, bl, bu, hs, xn, pi )

      implicit           real(8)          (a-h,o-z)
      integer            T
      integer            ha(maxne), hs(maxnb)
      integer            ka(maxn+1)
      real(8)            a(maxne) , bl(maxnb), bu(maxnb),
     $                   xn(maxnb), pi(maxm)

*     ------------------------------------------------------------------
*     spdata generates data for the "Spring" optimal control problem.
*     The constraints take the form
*              f(x) + A*x + s = 0,
*     where the Jacobian for f(x) + Ax is stored in a(*), and any
*     terms coming from f(x) are in the TOP LEFT-HAND CORNER of a(*),
*     with dimensions  nncon x nnjac.
*     Note that the right-hand side is zero.
*     s is a set of slack variables whose bounds contain any constants
*     that might have formed a right-hand side.
*
*     The objective function is
*             F(x) + c'x
*     where c would be row iobj of A (but there is no such row in
*     this example).  F(x) involves only the FIRST nnobj variables.
*
*     On entry,
*     T       is the number of time periods.
*     maxm, maxn, maxnb, maxne are upper limits on m, n, nb, ne.
*
*     On exit,
*     inform  is 0 if there is enough storage, 1 otherwise.
*     m       is the number of nonlinear and linear constraints.
*     n       is the number of variables.
*     nb      is n + m.
*     ne      is the number of nonzeros in a(*).
*     nncon   is the number of nonlinear constraints (they come first).
*     nnobj   is the number of nonlinear objective variables.
*     nnjac   is the number of nonlinear Jacobian variables.
*     a       is the constraint matrix (Jacobian), stored column-wise.
*     ha      is the list of row indices for each nonzero in a(*).
*     ka      is a set of pointers to the beginning of each column of a.
*     bl      is the lower bounds on x and s.
*     bu      is the upper bounds on x and s.
*     hs(1:n) is a set of initial states for each x  (0,1,2,3,4,5).
*     xn(1:n) is a set of initial values for x.
*     pi(1:m) is a set of initial values for the dual variables pi.
*
*     14 Nov 1994: First version of spdata.
*     ------------------------------------------------------------------

      parameter      ( zero   = 0.0d+0,   one    = 1.0d+0 )
      parameter      ( bplus  = 1.0d+20,  dummy  = 0.111111d+0 )

*     T defines the dimension of the problem.

      m      = T*2
      n      = T*3 + 2
      nb     = n   + m
      nncon  = T
      nnobj  = T*2 + 2  ! y(0:T) and x(0:T)
      nnjac  = T   + 1  ! y(0:T)
      ne     = T*7

*     Check if there is enough storage.

      inform = 0
      if (m      .gt. maxm ) inform = 1
      if (n      .gt. maxn ) inform = 1
      if (nb     .gt. maxnb) inform = 1
      if (ne     .gt. maxne) inform = 1
      if (inform .gt.   0  ) return

*     ------------------------------------------------------------------
*     Generate columns for y(t), t = 0 to T.
*     The first T rows are nonlinear, and the next T are linear.
*     The Jacobian is T x (T+1) upper bidiagonal.
*     We generate the sparsity pattern here.
*     We put in dummy numerical values for the nonlinear gradients.
*     The true non-constant values are computed by funcon.
*     ------------------------------------------------------------------
      j      = 0   ! counts the variables
      ne     = 0   ! counts the Jacobian and linear constraint entries

      do 100, k = 0, T
         j      =   j  + 1
         ka(j)  =   ne + 1
         bl(j)  = - one
         bu(j)  =   bplus
         xn(j)  = - one
         hs(j)  =   0      ! Make the y(t) nonbasic.

*        There are two Jacobian nonzeros per column,
*        except in the first and last column.

         if (k .gt. 0) then    !  Aij = 1
            ne     =   ne + 1
            ha(ne) =   k
            a(ne)  =   one
         end if

         if (k .lt. T) then    !  Aij = .02y - 1  (nonlinear)
            ne     =   ne + 1
            ha(ne) =   k  + 1
            a(ne)  =   dummy
         end if

*        Below the Jacobian the linear constraints are diagonal.

         if (k .lt. T) then
            ne     =   ne + 1
            ha(ne) =   T  + k + 1
            a(ne)  = - 0.2d+0
         end if
  100 continue

*     ------------------------------------------------------------------
*     Generate columns for x(t), t = 0 to T.
*     They form 0.004*I in the first T rows,
*     and an upper-bidiagonal in the last T rows.
*     ------------------------------------------------------------------
      do 200, k = 0, T
         j      =   j  + 1
         ka(j)  =   ne + 1
         bl(j)  = - bplus
         bu(j)  =   bplus
         xn(j)  =   zero
         hs(j)  =   3     ! Make the x(t) basic.

*        Part of 0.004*I.

         if (k .lt. T) then
            ne     =   ne + 1
            ha(ne) =   k  + 1
            a(ne)  =   0.004d+0
         end if

*        The bidiagonal parts have two entries
*        except in the first and last columns.

         if (k .gt. 0) then    !  Aij = 1
            ne     =   ne + 1
            ha(ne) =   T  + k
            a(ne)  =   one
         end if

         if (k .lt. T) then    !  Aij = - 1
            ne     =   ne + 1
            ha(ne) =   T  + k + 1
            a(ne)  = - one
         end if
  200 continue

*     ------------------------------------------------------------------
*     Generate columns for u(t), t = 0 to T-1.
*     They form -0.2I in the first T rows.
*     ------------------------------------------------------------------
      do 300, k = 0, T - 1
         j      =   j  + 1
         ka(j)  =   ne + 1
         bl(j)  = - 0.2d+0
         bu(j)  =   0.2d+0
         xn(j)  =   zero
         hs(j)  =   3     ! Make the u(t) basic.

         ne     =   ne + 1
         ha(ne) =   k  + 1
         a(ne)  = - 0.2d+0
  300 continue

*     ka(*) has one extra element.
*     Some of the variables are fixed.

      ka(n+1) = ne + 1
      bl(1)   = zero      ! y(0) = 0
      bu(1)   = zero
      bl(T+1) = zero      ! y(T) = 0
      bu(T+1) = zero
      bl(T+2) = 10.0d+0   ! x(0) = 10
      bu(T+2) = 10.0d+0

*     ------------------------------------------------------------------
*     Set bounds on the slacks.
*     We don't need to set initial values and states for slacks
*     (assuming MINOS does a cold start).
*     ------------------------------------------------------------------
      do 500, j = n + 1, nb
         bl(j)  = zero
         bu(j)  = zero
  500 continue

*     Initialize pi.
*     MINOS requires only pi(1:nncon) to be initialized.
*     We initialize all of pi just in case.

      do 600,  i = 1, T
         pi(i)   =   zero
         pi(T+i) =   zero
  600 continue

*     end of spdata
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine funobj( mode, n, x, f, g, nstate, nprob, z, lenz )

      implicit           real(8)          (a-h,o-z)
      real(8)            x(n), g(n), z(lenz)

*     ------------------------------------------------------------------
*     This is funobj for problem Spring  (an optimal control problem).
*     ------------------------------------------------------------------

      integer            T
      parameter        ( zero = 0.0d+0 )

      T      = (n - 2)/2
      f      = zero
      jy     = 0
      jx     = T + 1

      do 50, k = 0, T
         jy    = jy + 1
         jx    = jx + 1
         u     = x(jx)
         f     = f  +  u**2
         g(jy) = zero
         g(jx) = u
   50 continue

      f = f / 2.0d+0

*     end of funobj for Spring
      end

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine funcon( mode, m, n, njac, x, f, g,
     $                   nstate, nprob, z, lenz )

      implicit           real(8)          (a-h,o-z)
      real(8)            x(n), f(m), g(njac), z(lenz)

*     ------------------------------------------------------------------
*     This is funcon for problem Spring  (Optimal Control Problem).
*     The Jacobian is upper bidiagonal,
*     and only the diagonal terms are nonlinear.
*     The constant 1's in the Jacobian are not regenerated here.
*     ------------------------------------------------------------------

      integer            T
      parameter        ( one = 1.0d+0 )

      T     = n - 1
      jy    = 0      ! Counts y(t) variables
      jg    = - 1    ! Counts nonlinear Jacobian elements

      do 150, i = 1, T
         jy     = jy + 1
         jg     = jg + 2
         yt     = x(jy)
         ytp1   = x(jy + 1)
         f(i)   = 0.01d+0 * yt**2  +  (ytp1  -  yt)
         g(jg)  = 0.02d+0 * yt               -  one
*--      g(jg+1)= one      ! Constant element set by spdata.
  150 continue

      return

      entry matmod

*     end of funcon for Spring
      end
