*     ------------------------------------------------------------------
*     File ramsey.f
*     This is a main program to test subroutine minoss, which is
*     part of MINOS 5.5.  It generates the Ramsey economic model called
*     MANNE on Pages 98-108 of the MINOS 5.1 User's Guide, then asks
*     minoss to solve it.
*
*     
*     11 Nov 1991: First version.  File manne.for.
*     27 Nov 1991: miopt, miopti, mioptr used to alter some options
*                  for a second call to minoss.
*     10 Apr 1992: objadd added as input parameter to minoss.
*     26 Jun 1992: integer*2 changed to integer*4.
*     12 Nov 1993: Generalized to read nt as "Nonlinear constraints"
*                  from Specs file.
*     07 Feb 1997: Jacobian = Sparse  no longer needed by minoss.
*     14 Sep 2000: Renamed ramsey.f.  Converted to use mistart.
*     ------------------------------------------------------------------

      program            ramsey

      implicit           double precision (a-h,o-z)

      common    /m8len / njac  ,nncon ,nncon0,nnjac

      parameter        ( maxm   = 10000,
     $                   maxn   = 15000,
     $                   maxnb  = maxm + maxn,
     $                   maxne  = 30000,
     $                   nname  = 1 )

      character*8        names(5)  , probnm
      integer            ha(maxne) , hs(maxnb)
      integer            ka(maxn+1), name1(nname), name2(nname)
      double precision   a(maxne)  , bl(maxnb)   , bu(maxnb),
     $                   xn(maxnb) , pi(maxm)    , rc(maxnb)

      parameter          ( lenz = 500000 ) ! As big as you like.
      double precision   z(lenz)           ! This is the MINOS workspace.
*     ------------------------------------------------------------------

*     Give names to the Problem, Objective, Rhs, Ranges and Bounds.

      names(1) = 'manne...'
      names(2) = 'funobj  '
      names(3) = 'zero    '
      names(4) = 'range1  '
      names(5) = 'bound1  '


!     Specify some file numbers.  (Others may be in the SPECS file.)
!     0 means that there should be no file.

      iprint = 9   ! The MINOS PRINT   file.
      isumm  = 6   ! The MINOS SUMMARY file.
      ispecs = 4   ! The MINOS SPECS   file.
      nout   = 6   ! Local output file (> 0).

!     ------------------------------------------------------------------
!     Now we may open any number of files ourselves
!     (perhaps to give them sensible names).
!     For example:
!
      open( iprint, file='ramsey.out', status='UNKNOWN')
      open( ispecs, file='ramsey.spc', status='OLD')
!
!     Alternatively, we may let mistart and minoss open them,
!     using the method selected in subroutine m1open
!     in the mi10*.f file that was used to build MINOS.
!
!     RULE OF THUMB:
!     MINOS won't open file ispec (for example)
!     if that unit is already open, or if ispec = 0.
!     ------------------------------------------------------------------


!     ------------------------------------------------------------------
!     mistart MUST BE CALLED BEFORE ANY OTHER MINOS ROUTINE.
!     ------------------------------------------------------------------
      call mistart( iprint, isumm, ispecs )  ! Initialize MINOS and open
                                             ! the specified files.

      call mispec( ispecs, inform )          ! Read the SPECS file
                                             ! (if ispecs > 0).

      if (inform .ge. 2) then
         write(nout, *) 'Error: ispecs > 0 but no SPECS file found'
         stop
      end if


!     ------------------------------------------------------------------
!     Generate an nt-period problem.
!     ------------------------------------------------------------------
      nt     = nncon
      if (nt .le. 1  .or.  nt .gt. maxm/2) then
         write(nout, *) 'Invalid  nt  specified:', nt
         stop
      end if

*     Write nt into the problem name.

      write(probnm, '(i8)') nt
      if      (nt .lt.  1000) then
         probnm(1:5) = 'ramse'
      else if (nt .lt. 10000) then
         probnm(1:4) = 'rams'
      else
         probnm(1:3) = 'ram'
      end if

      names(1) = probnm
      write(nout, *) '      T =', nt

*     Generate the constraint data.

      call t4data( nt, maxm, maxn, maxnb, maxne, inform,
     $             m, n, nb, ne, nncon, nnobj, nnjac,
     $             a, ha, ka, bl, bu, hs, xn, pi )

      if (inform .ge. 1) then
         write(nout, *) 'Not enough storage to generate a problem ',
     $                  'with  nt =', nt
         stop
      end if             

*     ------------------------------------------------------------------
*     Specify options that were not set in the Specs file.
*     i1 and i2 may refer to the Print and Summary file respectively.
*     Setting them to 0 suppresses printing.
*     ------------------------------------------------------------------
      maxr   =   2*nt + 1
      itnlim = 100*nt
      i0     = 0
      i1     = nout
      i2     = 0
      call miopti( 'Problem number    ',   1114, i0, i2, inform )
      call miopti( 'Hessian dimension ', maxr  , i1, i2, inform )
      call miopti( 'Iterations        ', itnlim, i1, i2, inform )

!     ------------------------------------------------------------------
!     Solve the problem, using a Cold start.
!     iobj   = 0   means there is no linear objective row in a(*).
!     objadd = 0.0 means there is no constant to be added to the objective.
!
!     t4data has defined various dimensions and data arrays
!     and also initialized hs, xn, pi.
!     nname  = 1   means t4data didn't try to generate meaningful names
!     (for the variables and constraints) inside name1(*) and name2(*).
!     MINOS will print default names.
!     ------------------------------------------------------------------
      iobj   = 0
      objadd = 0.0d+0
      nwcore = lenz

      call minoss( 'Cold', m, n, nb, ne, nname,
     $             nncon, nnobj, nnjac,
     $             iobj, objadd, names,
     $             a, ha, ka, bl, bu, name1, name2,
     $             hs, xn, pi, rc, 
     $             inform, mincor, ns, ninf, sinf, obj,
     $             z, nwcore )

      if (inform .eq. 42) then
         write(nout, *) ' '
         write(nout, *) 'Estimate of required nwcore:', mincor
         stop
      end if

      write(nout, *) ' '
      write(nout, *) 'minoss finished.'
      write(nout, *) 'inform =', inform
      if (inform .ge. 20) go to 900
*  Stop anyway
      if (inform .lt. 20) stop


*     ------------------------------------------------------------------
*     Test the Hot start.
*     The following illustrates the use of miopt, miopti and mioptr
*     to set specific options.  If necessary, we could ensure that
*     all unspecified options take default values
*     by first calling miopt ( 'Defaults', ... ).
*     Beware that certain parameters would then need to be redefined.
*     ------------------------------------------------------------------
      write(nout, *) ' '
      write(nout, *) 'Test Hot start without scaling.'

      inform = 0
      penpar = 0.01
      call miopt ( '                  ',         iprint, isumm, inform )
*---  call miopt ( 'Defaults          ',         iprint, isumm, inform )
*---  call miopti( 'Problem number    ',   1114, iprint, isumm, inform )
*---  call miopt ( 'Maximize          ',         iprint, isumm, inform )
*---  call miopt ( 'Jacobian    Sparse',         iprint, isumm, inform )
      call miopt ( 'Derivative level 3',         iprint, isumm, inform )
      call miopt ( 'Print  level     1',         iprint, isumm, inform )
      call miopt ( 'Print  frequency 1',         iprint, isumm, inform )
      call miopt ( 'Verify level     0',         iprint, isumm, inform )
      call miopt ( 'Scale option     0',         iprint, isumm, inform )
      call mioptr( 'Penalty parameter ', penpar, iprint, isumm, inform )

      call miopt ( 'Completion Full   ',         iprint, isumm, inform )
      call mioptr( 'Optimality tol    ', 1.0d-7, iprint, isumm, inform )

      if (inform .gt. 0) then
         write(nout, *) 'NOTE: Some of the options were not recognized'
      end if

*     Warm and Hot starts are normally used after minoss has solved a
*     problem with the SAME DIMENSIONS but perhaps altered data.
*     As with a Warm start, hs(*) specifies a basis from the
*     previous call.  In addition, up to three items from the previous
*     call can be reused.  They are denoted by F, H and S as follows:
*     'Hot F'    means use the existing basis FACTORS (B = LU).
*     'Hot H'    means use the existing reduced HESSIAN approximation.
*     'Hot S'    means use the existing column and row SCALES.
*     'Hot FS'   means use the Factors and Scales but not the Hessian.
*     'Hot FHS'  means use all three items.
*     'Hot'      is equivalent to 'Hot FHS'.
*     The letters F,H,S may be in any order.

      call minoss( 'Hot FH', m, n, nb, ne, nname,
     $             nncon, nnobj, nnjac,
     $             iobj, objadd, names,
     $             a, ha, ka, bl, bu, name1, name2,
     $             hs, xn, pi, rc, 
     $             inform, mincor, ns, ninf, sinf, obj,
     $             z, nwcore )

      write(nout, *) ' '
      write(nout, *) 'minoss Hot Start finished.'
      write(nout, *) 'inform =', inform
      write(nout, *) 'obj    =', obj
      if (inform .ge. 20) go to 900

*     ------------------------------------------------------------------
*     Test the Warm start.
*     ------------------------------------------------------------------
      write(nout, *) ' '
      write(nout, *) 'Test Warm start with scaling.'

      call miopt ( 'Scale option     2',        iprint, isumm, inform )
      call mioptr( 'Optimality tol    ', 1.0d-7,iprint, isumm, inform )
      call miopt ( 'Completion Full   ',        iprint, isumm, inform )

      call minoss( 'Warm', m, n, nb, ne, nname,
     $             nncon, nnobj, nnjac,
     $             iobj, objadd, names,
     $             a, ha, ka, bl, bu, name1, name2,
     $             hs, xn, pi, rc, 
     $             inform, mincor, ns, ninf, sinf, obj,
     $             z, nwcore )

      write(nout, *) ' '
      write(nout, *) 'minoss Warm Start finished.'
      write(nout, *) 'inform =', inform
      write(nout, *) 'obj    =', obj
      if (inform .ge. 20) go to 900
      stop

*     ------------------------------------------------------------------
*     Error exit.
*     ------------------------------------------------------------------
  900 write(nout, *) ' '
      write(nout, *) 'STOPPING because of error condition'
      stop

      end ! program ramsey

*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

      subroutine t4data( nt, maxm, maxn, maxnb, maxne, inform,
     $                   m, n, nb, ne, nncon, nnobj, nnjac,
     $                   a, ha, ka, bl, bu, hs, xn, pi )

      implicit           double precision (a-h,o-z)
      integer*4          ha(maxne), hs(maxnb)
      integer            ka(maxn+1)
      double precision   a(maxne) , bl(maxnb), bu(maxnb),
     $                   xn(maxnb), pi(maxm)

*     ------------------------------------------------------------------
*     t4data  generates data for the test problem t4manne
*     (called problem MANNE in the MINOS 5.1 User's Guide).
*     The constraints take the form
*              f(x) + A*x + s = 0,
*     where the Jacobian for f(x) + Ax is stored in a(*), and any
*     terms coming from f(x) are in the TOP LEFT-HAND CORNER of a(*),
*     with dimensions  nncon x nnjac.
*     Note that the right-hand side is zero.
*     s is a set of slack variables whose bounds contain any constants
*     that might have formed a right-hand side.
*
*     The objective function is
*             F(x) + c'x
*     where c would be row iobj of A (but there is no such row in
*     this example).  F(x) involves only the FIRST nnobj variables.
*   
*     On entry,
*     nt      is T, the number of time periods.
*     maxm, maxn, maxnb, maxne are upper limits on m, n, nb, ne.
*
*     On exit,
*     inform  is 0 if there is enough storage, 1 otherwise.
*     m       is the number of nonlinear and linear constraints.
*     n       is the number of variables.
*     nb      is n + m.
*     ne      is the number of nonzeros in a(*).
*     nncon   is the number of nonlinear constraints (they come first).
*     nnobj   is the number of nonlinear objective variables.
*     nnjac   is the number of nonlinear Jacobian variables.
*     a       is the constraint matrix (Jacobian), stored column-wise.
*     ha      is the list of row indices for each nonzero in a(*).
*     ka      is a set of pointers to the beginning of each column of a.
*     bl      is the lower bounds on x and s.
*     bu      is the upper bounds on x and s.
*     hs(1:n) is a set of initial states for each x  (0,1,2,3,4,5).
*     xn(1:n) is a set of initial values for x.
*     pi(1:m) is a set of initial values for the dual variables pi.
*
*     09 Jul 1992: No need to initialize xn and hs for the slacks.
*     15 Oct 1993: pi is now an output parameter.  (Should have been
*                  all along.)
*     09 Aug 1995: Initialize Jacobian variables to be nonbasic
*                  between their bounds -- not superbasic.
*     ------------------------------------------------------------------

      parameter      ( zero   = 0.0d+0,   one    = 1.0d+0,
     $                 dummy  = 0.1d+0,   growth = .03d+0,
     $                 bplus  = 1.0d+20,  bminus = - bplus )

*     nt defines the dimension of the problem.

      m      = nt*2
      n      = nt*3
      nb     = n + m
      nncon  = nt
      nnobj  = nt*2
      nnjac  = nt
      ne     = nt*6 - 1
      T      = nt

*     Check if there is enough storage.

      inform = 0
      if (m      .gt. maxm ) inform = 1
      if (n      .gt. maxn ) inform = 1
      if (nb     .gt. maxnb) inform = 1
      if (ne     .gt. maxne) inform = 1
      if (inform .gt.   0  ) return

*     Generate columns for Capital (Kt, t = 1 to nt).
*     The first nt rows are nonlinear, and the next nt are linear.
*     The Jacobian is an nt x nt diagonal.
*     We generate the sparsity pattern here.
*     We put in dummy numerical values of 0.1 for the gradients.
*     Real values for the gradients are computed by t4con.

      ne     = 0
      do 100  k = 1, nt

*        There is one Jacobian nonzero per column.

         ne     = ne + 1
         ka(k)  = ne
         ha(ne) = k
         a(ne)  = dummy

*        The linear constraints form an upper bidiagonal pattern.

         if (k .gt. 1) then
            ne     = ne + 1
            ha(ne) = nt + k - 1
            a(ne)  = one
         end if

         ne     = ne + 1
         ha(ne) = nt + k
         a(ne)  = - one
  100 continue

*     The last nonzero is special.

      a(ne)  = growth

*     Generate columns for Consumption (Ct for t = 1 to nt).
*     They form -I in the first nt rows.
*     jC and jI are base indices for the Ct and It variables.

      jC    = nt
      jI    = nt*2

      do 200 k = 1, nt
         ne       = ne + 1
         ka(jC+k) = ne
         ha(ne)   = k
         a(ne)    = - one
  200 continue

*     Generate columns for Investment (It for t = 1 to nt).
*     They form -I in the first nt rows and -I in the last nt rows.

      do 300 k = 1, nt
         ne       = ne + 1
         ka(jI+k) = ne
         ha(ne)   = k
         a(ne)    = - one
         ne       = ne + 1
         a(ne)    = - one
         ha(ne)   = nt + k
  300 continue

*     ka(*) has one extra element.

      ka(n+1) = ne + 1

*     Set lower and upper bounds for Kt, Ct, It.
*     Also initial values and initial states for all variables.
*     The Jacobian variables are the most important.
*     We used to make them all superbasic.
*     Now they are nonbasic between their bounds.
*     The others are ok nonbasic.
*     For test purposes, we want the initial x to be infeasible
*     with respect to the linear constraints.
*     Try setting the last Kapital too high.


      do 400  k = 1, nt
         bl(   k) = 3.05d+0
         bu(   k) = bplus
         bl(jC+k) = 0.95d+0
         bu(jC+k) = bplus
         bl(jI+k) = 0.05d+0
         bu(jI+k) = bplus

         xn(   k) = 3.0d+0 + (k - 1)/10.0d+0
         xn(jC+k) = bl(jC+k)
         xn(jI+k) = bl(jI+k)

         hs(   k) = 0
         hs(jC+k) = 0
         hs(jI+k) = 0

         if (k .eq. nt) then
            xn(k) = 1.0d+3
            hs(k) = 2
         end if
  400 continue

*     The first Capital is fixed.
*     The last three Investments are bounded.
*     Fudge them to be the normal ones for T = 10.

      scale       = T / 10.0
      bu(1)       = bl(1)
      xn(1)       = bl(1)
      hs(1)       = 0
      bu(jI+nt-2) = 0.112d+0 * scale
      bu(jI+nt-1) = 0.114d+0 * scale
      bu(jI+nt  ) = 0.116d+0 * scale

*     Set bounds on the slacks.
*     The nt nonlinear (Money)    rows are >=.
*     The nt    linear (CapacitY) rows are <=.
*     We no longer need to set initial values and states for slacks
*     (assuming MINOS does a cold start).

      jM     = n
      jY     = n + nt

      do 500    k = 1, nt
         bl(jM+k) = bminus
         bu(jM+k) = zero
         bl(jY+k) = zero
         bu(jY+k) = bplus

*-       xn(jM+k) = zero
*-       xn(jY+k) = zero
*-       hs(jM+k) = 0
*-       hs(jY+k) = 0
  500 continue

*     The last Money and Capacity rows have a Range.

      bl(jM+nt) = - 10.0d+0
      bu(jY+nt) =   20.0d+0

*     Initialize pi.
*     MINOS requires only pi(1:nncon) to be initialized.
*     We initialize all of pi just in case.

      do 600 i = 1, nt
         pi(i)    = - one
         pi(nt+i) = + one
  600 continue

      end ! subroutine t4data
