function writeMPS( modelNo )
%        writeMPS( modelNo )
% where modelNo refers to one of the following data files and associated output:
%
% modelNo              Input file                           Output file
%   1  ../data/FBA/model_final_build_unscaled.mat           TMA.mps
%   2  ../data/FBA/modelO.mat                               model0.mps
%   4  ../data/FBA/ModelGlcAer_WT.mat     % upper-case M    GlcAerWT.mps
%   5  ../data/FBA/modelGlcAer_WTlift.mat % lower-case m    GlcAlift.mps
%   6  ../data/FBA/10_01_27_ME_for_Michael.mat              10_01_ME.mps
%  19  ../data/FBA/model_Recon1.mat
%
% Beware that the .mat files have a few critical differences.
% writeMPS.m tests for them.

% 24 Mar 2014: dumpMPS.m derived from dumpLP.m.
%              Ding Ma and Michael Saunders, SOL, Stanford University.
% 29 Apr 2014: dumpMPS.m added to quadLP distribution.
% 05 Sep 2016: Ambros Gleixner advised that lower bounds of -Inf were
%              being written to the MPS file as -Inf (instead of -1E+20 say).
%              dumpMPS.m treated xl(j) = 0.0 and xu(j) = +Inf specially,
%              but we forgot about xl(j) = -Inf.
%              By chance Minos didn't mind, but SoPlex did!
%              -Inf is nonstandard, so we must change it.
%              Ambros mentioned also that spaces in names cause trouble.
%              He suggests "_".
% 13 Sep 2016: writeMPS.m replaces dumpMPS.m.
%              Bounds may include FX, FR, MI+UP, LO, UP.
%              Row and columns names are now of the form r0000001 and x0000001
%              (taking advantage of %07g format).
% 15 Sep 2016: Changed writeMPS.m script into function writeMPS( modelNo )
%              for a few specific values of modelNo (which can be extended).
%
%              Note that the Cobra toolbox contains BuildMPS.m, but
%              this requires Ax = b and Aeq x = beq to be separate,
%              rather than having csense for each individual constraint.

switch modelNo
% case 0  % Josh's model from Dec 14  2010
%   load ../data/FBA/modelFinal.mat
%   mname = 'TMAfinal';
%   fname = 'TMAfinal.mps'

  case 1   % Josh's model    
    dname = '../data/FBA/model_final_build_unscaled.mat';
    load(dname) 
    mname = 'TMA_ME';
    fname = 'TMA_ME.mps';

  case 2   % Ronan and Ines's data  
    dname = '../data/FBA/modelO.mat';
    load(dname)
    model = modelO;
    model.S = model.A;                %%%%%%%%% Special to this problem
    clear   modelO;
    mname = 'modelO';
    fname = 'modelO.mps'

% case 3   % Ronan and Ines's data
% load ~/Dropbox/linearfba/data/iAF1260_pathological_optPerturbation_matrix.mat
% This loads only A 9158x9158
% fname = 'iAF1260.mps'

  case 4   % Ronan and Ines's data
    dname = '../data/FBA/ModelGlcAer_WT.mat'        % upper-case M
    load(dname)
    model = modelGlcAer_WT;                         % lower-case m
    clear   modelGlcAer_WT;
    mname = 'GlcAerWT'
    fname = 'GlcAerWT.mps'

  case 5   % Ronan and Ines's data with Yuekai's lifting
    dname = '../data/FBA/modelGlcAer_WTlift.mat';   % lower-case m
    load(dname)
    model = modelGlcAer_WTchn;                      % lower-case m
    model.S = model.A;                %%%%%%%%% Special to this problem
    clear   modelGlcAer_WTchn;
    mname = 'GlcAlift';
    fname = 'GlcAlift.mps'

  case 6   % Ronan and Ines's data
    dname = '../data/FBA/10_01_27_ME_for_Michael.mat';
    load(dname)
    model = modelGlycAer_WT_Adj;
    clear   modelGlycAer_WT_Adj;
    mname = '10_01_ME';
    fname = '10_01_ME.mps'

  case 19  % Recon1
    dname = '../data/FBA/model_Recon1.mat';
    load(dname)
    model = model_Recon1;
    mname = 'Recon1'; 
    fname = 'Recon1.mps'

  % case 20  % ThMa
  %   dname = 'data/model_ThMa.mat';
  %   load(dname)
  %   model = model_ThMa;
  %   mname = 'ThMa';
  %   fname = 'ThMa.mps';
    
  % case 21  % GlcAer
  %   dname = 'data/model_GlcAer.mat';
  %   load(dname)
  %   model = model_GlcAer;
  %   mname = 'GlcAer';
  %   fname = 'GlcAer.mps';

  otherwise
    disp('Unknown modelNo')
end

disp(['Loaded file  ' dname])
S      = model.S;
b      = model.b;
c      = model.c;
xl     = model.lb;
xu     = model.ub;
[m,n]  = size(S);

if isfield(model,'csense')
    csense = model.csense;      % E or L or G
else
    csense(1:m,1) = 'E';
end

if isfield(model,'osense')
    osense = model.osense;      % 1=min   (-1=max ?)
else
    osense = 1;
end

if modelNo==5
    osense  = 1;            %%%%%%%%% Special to this problem!
end

disp(['rows     ' num2str(m)])
disp(['columns  ' num2str(n)])
disp(['nonzeros ' num2str(nnz(S))])

% 06 Aug 2012: Include c as an additional free *last* row of Ax = b
%              so that SQOPT will treat the problem as an LP, not a QP.

disp('Sparse objective row c will become an added row of S')
S     = [      S            % Luckily we only have to do it once
         osense*sparse(c)'];
b     = [b; 0];
csense(m+1,1) = 'N';
[m,n] = size(S);

%----------------------------------------------------------------------
% Generate dummy row name and column names
% 12 Sep 2016: Format %7g changed to %07g to avoid spaces in names.
%              Row and column names are now like r0000001 and c0000001.
%----------------------------------------------------------------------
rname(1:m,1:8) = ' ';
for i=1:m
   rname(i,:) = sprintf('r%07g', i);
end

cname(1:n,1:8) = ' ';
for j=1:n
   cname(j,:) = sprintf('x%07g', j);
end

%----------------------------------------------------------------------
% Write MPS file.
%----------------------------------------------------------------------
bigbnd = 1e+20;          % Bigger than MINOS and SQOPT's "infinite" bound.
                         % Note that 1e+100 is infinity for SoPlex.
tic
fid = fopen(fname,'w');  % Open file "fname" for writing
fprintf(fid,'NAME          %c%c%c%c%c%c%c%c', mname);
fprintf(fid,'\n');       % Has to be separate from previous line

fprintf(fid,'ROWS\n');
for i=1:m
    fprintf(fid,'  %c %c%c%c%c%c%c%c%c\n', csense(i),rname(i,:));
end

fprintf(fid,'COLUMNS\n');
for j=1:n
   c = S(:,j);
   [I,J,V] = find(c);
   for k=1:length(I);
      i = I(k);
      fprintf(fid,'    %c%c%c%c%c%c%c%c  %c%c%c%c%c%c%c%c  %12.5e\n', ...
                       cname(j,:),       rname(i,:),       V(k));
   end
end

fprintf(fid,'RHS\n');
[I,J,V] = find(b);
for k=1:length(I)
   i = I(k);
   fprintf(fid, '    RHS       %c%c%c%c%c%c%c%c  %12.5e\n', rname(i,:), b(i));
end

fprintf(fid,'BOUNDS\n');
for j=1:n
   bl = xl(j);
   bu = xu(j);
   cn = cname(j,:);
   if bl == bu
       fprintf(fid, ' FX BND       %c%c%c%c%c%c%c%c  %12.5e\n', cn, bl);
   elseif bl <= -bigbnd
       if bu >= +bigbnd
          fprintf(fid, ' FR BND       %c%c%c%c%c%c%c%c\n', cn);
       else
          fprintf(fid, ' MI BND       %c%c%c%c%c%c%c%c\n', cn);
          fprintf(fid, ' UP BND       %c%c%c%c%c%c%c%c  %12.5e\n', cn, bu);
       end
   else
       if bl ~= 0.0
          fprintf(fid, ' LO BND       %c%c%c%c%c%c%c%c  %12.5e\n', cn, bl);
       end
       if bu < bigbnd
          fprintf(fid, ' UP BND       %c%c%c%c%c%c%c%c  %12.5e\n', cn, bu);
       end
   end
end

fprintf(fid,'ENDATA\n');
fclose(fid);
disp(['Created file  ' fname])
toc

disp(' ')
disp( 'MPS file stats:')
disp(['rows     ' num2str(m)])
disp(['columns  ' num2str(n)])
disp(['nonzeros ' num2str(nnz(S))])

% Categorize row-types as in MINOS.
bigL   = -bigbnd;
bigU   =  bigbnd;
nL     = length(find(csense=='L'));
nG     = length(find(csense=='G'));
nN     = length(find(csense=='N'));
nE     = length(find(csense=='E'));
nbndd  = 0;
normal = nL + nG;
disp(' ')
disp('Row types as in MINOS:')
disp('              Total      Normal        Free       Fixed     Bounded')
fprintf('Rows   %12i%12i%12i%12i%12i\n', m,normal,nN,nE,nbndd)

% Categorize bounds as in PDCO.
pos  =  find( xl==0    & xu>=bigU );
neg  =  find( xl<=bigL & xu==0    );
low  =  find( xl> bigL & xl< xu   );
upp  =  find( xu< bigU & xl< xu   );
two  =  find( xl> bigL & xu< bigU & xl< xu );
fix  =  find( xl==xu );
free =  find( xl<=bigL & xu>=bigU );
zlo  =  find( xl==0    & xl< xu   );
zup  =  find( xu==0    & xl< xu   );

disp(' ')
disp('Bound types as in PDCO:')
fprintf('  [0,inf]  [-inf,0]  Finite xl  Finite xu  Two bnds   Fixed    Free')
fprintf('\n %8g %9g %10g %10g %9g %7g %7g', ...
        length(pos), length(neg), length(low),  ...
        length(upp), length(two), length(fix), length(free))
fprintf('\n  [0, xu]  [xl,  0]  excluding fixed variables')
fprintf('\n %8g %9g\n', length(zlo), length(zup))

% end function writeMPS
