import numpy as np

# ===== Phantom Generation Functions =====

def create_circular_phantom(size=32, radius=8, center=None):
    # Test phanton - all-ones with a circular ROS.
    if center is None:
        center = (size // 2, size // 2)
    
    y, x = np.ogrid[:size, :size]
    dist_from_center = np.sqrt((x - center[1])**2 + (y - center[0])**2)
    
    mask = dist_from_center <= radius
    phantom = np.zeros((size, size))
    phantom[mask] = 1
    
    return phantom

def create_rectangular_phantom(size=32, rect_size=None, center=None):
    # Test phanton - all-ones with a rectangular ROS.
    if rect_size is None:
        rect_size = (size // 3, size // 3)
    
    if center is None:
        center = (size // 2, size // 2)
    
    height, width = rect_size
    
    start_y = center[0] - height // 2
    end_y = start_y + height
    start_x = center[1] - width // 2
    end_x = start_x + width
    
    phantom = np.zeros((size, size))
    phantom[start_y:end_y, start_x:end_x] = 1
    
    return phantom

def create_shepp_logan_phantom(size=128):
    # Creating simulation of Shepp-Logan phantom test image
    x, y = np.mgrid[-1:1:size*1j, -1:1:size*1j]
    
    # Create ellipses for the phantom
    ellipses = [
        {'center': [0, 0], 'axes': [0.69, 0.92], 'angle': 0, 'value': 1.0},
        {'center': [0, -0.0184], 'axes': [0.6624, 0.874], 'angle': 0, 'value': -0.8},
        {'center': [0.22, 0], 'axes': [0.11, 0.31], 'angle': -18, 'value': -0.2},
        {'center': [-0.22, 0], 'axes': [0.16, 0.41], 'angle': 18, 'value': -0.2},
        {'center': [0, 0.35], 'axes': [0.21, 0.25], 'angle': 0, 'value': 0.1},
        {'center': [0, 0.1], 'axes': [0.046, 0.046], 'angle': 0, 'value': 0.1},
        {'center': [0, -0.1], 'axes': [0.046, 0.046], 'angle': 0, 'value': 0.1},
        {'center': [-0.08, -0.605], 'axes': [0.046, 0.023], 'angle': 0, 'value': 0.1},
        {'center': [0.06, -0.605], 'axes': [0.023, 0.023], 'angle': 0, 'value': 0.1}
    ]
    
    phantom = np.zeros((size, size))
    for ellipse in ellipses:
        # Rotate / translate coordinates
        angle_rad = ellipse['angle'] * np.pi / 180
        xr = (x - ellipse['center'][0]) * np.cos(angle_rad) - (y - ellipse['center'][1]) * np.sin(angle_rad)
        yr = (x - ellipse['center'][0]) * np.sin(angle_rad) + (y - ellipse['center'][1]) * np.cos(angle_rad)
        
        inside = ((xr / ellipse['axes'][0])**2 + (yr / ellipse['axes'][1])**2) <= 1.0
        phantom[inside] += ellipse['value']
    
    return phantom
    
def add_image_to_phantom(phantom, intensity=1):
    # Add an image to the phantom to simulate a nontrivial image within the ROS.
    
    size = phantom.shape[0]
    y, x = np.meshgrid(np.linspace(-1, 1, size), np.linspace(-1, 1, size), indexing='ij')
    
    # Use simulated Shepp-Logan phantom
    pattern = intensity * create_shepp_logan_phantom(size=size)
    
    # Apply the pattern only within the ROS
    result = pattern * phantom
    #print(np.mean(result[phantom==1]))
    #result = pattern
    return result
